<?php

namespace App\Plugins\Platforms\AndroidWebView\Livewire;

use App\Models\App;
use App\Models\PushNotificationConfig;
use Livewire\Component;
use Livewire\WithFileUploads;

class AppNotificationsSettings extends Component
{
    use WithFileUploads;

    public App $app;

    public $google_services_file;

    public $google_services_file_upload;

    public $firebase_credentials_file;

    public $firebase_credentials_file_upload;

    public $enabled;

    public function mount()
    {
        $config = $this->app->pushNotificationConfig;

        if ($config) {
            $this->google_services_file = $config->google_services_file;
            $this->firebase_credentials_file = $config->firebase_credentials_file;
            $this->enabled = $config->enabled;
        } else {
            $this->enabled = false;
        }
    }

    protected function rules()
    {
        return [
            'google_services_file_upload' => ['nullable', 'file', 'mimes:json', 'max:1024'],
            'firebase_credentials_file_upload' => ['nullable', 'file', 'mimes:json', 'max:2048'],
            'enabled' => ['required', 'boolean'],
        ];
    }

    public function save()
    {
        $this->validate();

        try {
            $data = [
                'enabled' => $this->enabled,
            ];

            // Handle Firebase Credentials file upload (Service Account JSON)
            if ($this->firebase_credentials_file_upload) {
                // Validate JSON syntax and structure
                $content = file_get_contents($this->firebase_credentials_file_upload->getRealPath());
                $json = json_decode($content, true);

                if (json_last_error() !== JSON_ERROR_NONE) {
                    $this->dispatch('show-toast',
                        type: 'error',
                        message: __('platform.toast.invalid_firebase_service_account')
                    );

                    return;
                }

                // Verify it's a valid service account file
                if (! isset($json['type']) || $json['type'] !== 'service_account') {
                    $this->dispatch('show-toast',
                        type: 'error',
                        message: __('platform.toast.invalid_firebase_service_account_type')
                    );

                    return;
                }

                $path = $this->firebase_credentials_file_upload->store('fcm/'.$this->app->id, 'private');
                $data['firebase_credentials_file'] = $path;
                $this->firebase_credentials_file = $path;
            }

            // Handle Google Services file upload
            if ($this->google_services_file_upload) {
                // Validate JSON syntax and structure
                $content = file_get_contents($this->google_services_file_upload->getRealPath());
                $json = json_decode($content, true);

                if (json_last_error() !== JSON_ERROR_NONE) {
                    $this->dispatch('show-toast',
                        type: 'error',
                        message: __('platform.toast.invalid_google_services_json')
                    );

                    return;
                }

                // Verify required keys exist
                if (! isset($json['project_info']) || ! isset($json['client'])) {
                    $this->dispatch('show-toast',
                        type: 'error',
                        message: __('platform.toast.invalid_google_services_structure')
                    );

                    return;
                }

                $path = $this->google_services_file_upload->store('fcm/'.$this->app->id, 'private');
                $data['google_services_file'] = $path;
                $this->google_services_file = $path;
            }

            // Create or update config
            PushNotificationConfig::updateOrCreate(
                ['app_id' => $this->app->id],
                $data
            );

            $this->reset(['google_services_file_upload', 'firebase_credentials_file_upload']);

            $this->dispatch('show-toast',
                type: 'success',
                message: __('platform.toast.notifications_saved')
            );

        } catch (\Exception $e) {
            $this->dispatch('show-toast',
                type: 'error',
                message: __('platform.toast.settings_save_failed')
            );
        }
    }

    public function deleteFirebaseCredentials()
    {
        try {
            $config = $this->app->pushNotificationConfig;

            if ($config && $config->firebase_credentials_file) {
                \Storage::disk('private')->delete($config->firebase_credentials_file);

                $config->update(['firebase_credentials_file' => null]);
                $this->firebase_credentials_file = null;

                $this->dispatch('show-toast',
                    type: 'success',
                    message: __('platform.toast.firebase_credentials_deleted')
                );
            }
        } catch (\Exception $e) {
            $this->dispatch('show-toast',
                type: 'error',
                message: __('platform.toast.firebase_credentials_delete_failed')
            );
        }
    }

    public function deleteGoogleServices()
    {
        try {
            $config = $this->app->pushNotificationConfig;

            if ($config && $config->google_services_file) {
                \Storage::disk('private')->delete($config->google_services_file);

                $config->update(['google_services_file' => null]);
                $this->google_services_file = null;

                $this->dispatch('show-toast',
                    type: 'success',
                    message: __('platform.toast.google_services_deleted')
                );
            }
        } catch (\Exception $e) {
            $this->dispatch('show-toast',
                type: 'error',
                message: __('platform.toast.google_services_delete_failed')
            );
        }
    }

    public function render()
    {
        return view('plugins.android-webview::livewire.app-notifications-settings');
    }
}
